/*:
 * @target MZ
 * @plugindesc v1.0 HS_FastMode連携：高速/自動/保護をスイッチでも制御＆同期＋コマンド
 * @author HS
 *
 * @help HS_FastMode_SwitchBridge.js
 * 【目的】既存のHS_FastModeの各設定（高速モード/自動送り/重要シーン保護）を
 * スイッチでも操作できるようにし、オプションとの双方向同期を提供します。
 * HS_FastMode.js の“下”に配置してください。
 *
 * ▼想定連携（HS_FastMode側のキー）
 *   ConfigManager.fastMode   … 高速モード
 *   ConfigManager.hsAuto     … 自動送り
 *   ConfigManager.hsProtect  … 重要シーン保護（有効/無効）
 *
 * @param FastModeSwitchId
 * @text 高速モード連動スイッチID
 * @type switch
 * @default 0
 * @desc >0 で有効。ON=高速ON / OFF=高速OFF。オプションと自動同期します。
 *
 * @param AutoSwitchId
 * @text 自動送り連動スイッチID
 * @type switch
 * @default 0
 * @desc >0 で有効。ON=自動ON / OFF=自動OFF。オプションと自動同期します。
 *
 * @param ProtectSwitchId
 * @text 保護（重要シーン）連動スイッチID
 * @type switch
 * @default 0
 * @desc >0 で有効。ON=保護ON / OFF=保護OFF。オプションと自動同期します。
 *
 * @param RealtimeSync
 * @text リアルタイム同期
 * @type boolean
 * @default true
 * @desc true: 毎フレーム双方向同期（楽） / false: コマンド時のみ同期（負荷最小）
 *
 * @param SyncOnMapStart
 * @text マップ開始時に同期
 * @type boolean
 * @default true
 * @desc マップ突入時に【スイッチ→オプション】へ一括反映します。
 *
 * @command SetFast
 * @text 高速モード設定
 * @arg state
 * @text 状態
 * @type select
 * @option ON @value on
 * @option OFF @value off
 * @option TOGGLE @value toggle
 *
 * @command SetAuto
 * @text 自動送り設定
 * @arg state
 * @text 状態
 * @type select
 * @option ON @value on
 * @option OFF @value off
 * @option TOGGLE @value toggle
 *
 * @command SetProtect
 * @text 保護（重要シーン）設定
 * @arg state
 * @text 状態
 * @type select
 * @option ON @value on
 * @option OFF @value off
 * @option TOGGLE @value toggle
 *
 * @command ApplyFromSwitches
 * @text スイッチ→オプションへ反映
 *
 * @command ApplyToSwitches
 * @text オプション→スイッチへ反映
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "HS_FastMode_SwitchBridge";
  const P = PluginManager.parameters(PLUGIN_NAME);
  const SW_FAST = Number(P.FastModeSwitchId || 0);
  const SW_AUTO = Number(P.AutoSwitchId || 0);
  const SW_PROT = Number(P.ProtectSwitchId || 0);
  const REALTIME = String(P.RealtimeSync || "true") === "true";
  const SYNC_ON_MAP = String(P.SyncOnMapStart || "true") === "true";

  // ---- safety: if HS_FastMode not loaded, still create keys
  if (ConfigManager.fastMode === undefined) ConfigManager.fastMode = false;
  if (ConfigManager.hsAuto   === undefined) ConfigManager.hsAuto   = false;
  if (ConfigManager.hsProtect=== undefined) ConfigManager.hsProtect= true;

  function cfg(key){ return !!ConfigManager[key]; }
  function setCfg(key, v){
    ConfigManager[key] = !!v;
    try { ConfigManager.save(); } catch(e) {}
  }

  const HSPrev = {
    sw_fast: null, sw_auto: null, sw_prot: null,
    cfg_fast: cfg("fastMode"), cfg_auto: cfg("hsAuto"), cfg_prot: cfg("hsProtect")
  };

  function readSw(id){ return id>0 ? !!$gameSwitches.value(id) : null; }
  function writeSw(id, v){ if(id>0) $gameSwitches.setValue(id, !!v); }

  function syncOne(id, key, prevSwKey, prevCfgKey){
    if (id <= 0) return;
    const swNow = readSw(id);
    const cfgNow = cfg(key);

    // 1) スイッチ側の変化を優先反映 → コンフィグへ
    if (swNow !== HSPrev[prevSwKey]) {
      setCfg(key, swNow);
      HSPrev[prevCfgKey] = cfg(key);
      HSPrev[prevSwKey] = swNow;
      return;
    }
    // 2) オプション側の変化をスイッチへ
    if (cfgNow !== HSPrev[prevCfgKey]) {
      writeSw(id, cfgNow);
      HSPrev[prevCfgKey] = cfgNow;
      HSPrev[prevSwKey] = readSw(id);
    }
  }

  function applyFromSwitches(){
    if (!$gameSwitches) return;
    if (SW_FAST>0) setCfg("fastMode",  readSw(SW_FAST));
    if (SW_AUTO>0) setCfg("hsAuto",    readSw(SW_AUTO));
    if (SW_PROT>0) setCfg("hsProtect", readSw(SW_PROT));
    HSPrev.cfg_fast = cfg("fastMode");
    HSPrev.cfg_auto = cfg("hsAuto");
    HSPrev.cfg_prot = cfg("hsProtect");
    HSPrev.sw_fast = readSw(SW_FAST);
    HSPrev.sw_auto = readSw(SW_AUTO);
    HSPrev.sw_prot = readSw(SW_PROT);
  }

  function applyToSwitches(){
    if (!$gameSwitches) return;
    if (SW_FAST>0) writeSw(SW_FAST, cfg("fastMode"));
    if (SW_AUTO>0) writeSw(SW_AUTO, cfg("hsAuto"));
    if (SW_PROT>0) writeSw(SW_PROT, cfg("hsProtect"));
    HSPrev.cfg_fast = cfg("fastMode");
    HSPrev.cfg_auto = cfg("hsAuto");
    HSPrev.cfg_prot = cfg("hsProtect");
    HSPrev.sw_fast = readSw(SW_FAST);
    HSPrev.sw_auto = readSw(SW_AUTO);
    HSPrev.sw_prot = readSw(SW_PROT);
  }

  function toggle(key){ setCfg(key, !cfg(key)); }

  // ---- plugin commands
  PluginManager.registerCommand(PLUGIN_NAME, "SetFast", args=>{
    if (args.state==="on") setCfg("fastMode", true);
    else if (args.state==="off") setCfg("fastMode", false);
    else toggle("fastMode");
    applyToSwitches();
  });
  PluginManager.registerCommand(PLUGIN_NAME, "SetAuto", args=>{
    if (args.state==="on") setCfg("hsAuto", true);
    else if (args.state==="off") setCfg("hsAuto", false);
    else toggle("hsAuto");
    applyToSwitches();
  });
  PluginManager.registerCommand(PLUGIN_NAME, "SetProtect", args=>{
    if (args.state==="on") setCfg("hsProtect", true);
    else if (args.state==="off") setCfg("hsProtect", false);
    else toggle("hsProtect");
    applyToSwitches();
  });
  PluginManager.registerCommand(PLUGIN_NAME, "ApplyFromSwitches", ()=> applyFromSwitches());
  PluginManager.registerCommand(PLUGIN_NAME, "ApplyToSwitches",   ()=> applyToSwitches());

  // ---- map start: switch -> config
  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function(){
    _Scene_Map_start.call(this);
    if (SYNC_ON_MAP) applyFromSwitches();
  };

  // ---- realtime polling (lightweight)
  function bridgeSync(){
    syncOne(SW_FAST, "fastMode",  "sw_fast", "cfg_fast");
    syncOne(SW_AUTO, "hsAuto",    "sw_auto", "cfg_auto");
    syncOne(SW_PROT, "hsProtect", "sw_prot", "cfg_prot");
  }

  if (REALTIME){
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function(){ _Scene_Map_update.call(this); bridgeSync(); };
    const _Scene_Battle_update = Scene_Battle.prototype.update;
    Scene_Battle.prototype.update = function(){ _Scene_Battle_update.call(this); bridgeSync(); };
  }

  // ---- Public API for script calls / UI buttons
  window.HS_FM_Switch = {
    setFast: (b)=>{ setCfg("fastMode", !!b); applyToSwitches(); },
    toggleFast: ()=>{ toggle("fastMode"); applyToSwitches(); },
    setAuto: (b)=>{ setCfg("hsAuto", !!b); applyToSwitches(); },
    toggleAuto: ()=>{ toggle("hsAuto"); applyToSwitches(); },
    setProtect: (b)=>{ setCfg("hsProtect", !!b); applyToSwitches(); },
    toggleProtect: ()=>{ toggle("hsProtect"); applyToSwitches(); }
  };
// --- Options画面を閉じるときに、オプション→スイッチへ確実に反映する
const _Scene_Options_terminate = Scene_Options.prototype.terminate;
Scene_Options.prototype.terminate = function() {
  try { 
    // 直前に触ったオプション値をスイッチへ押し込み
    if (typeof applyToSwitches === "function") applyToSwitches();
  } catch(e) {}
  _Scene_Options_terminate.call(this);
};

// --- オプション項目が変更されるたび即時で連動（任意・より強固）
const _Window_Options_setConfigValue = Window_Options.prototype.setConfigValue;
Window_Options.prototype.setConfigValue = function(symbol, value) {
  _Window_Options_setConfigValue.call(this, symbol, value);
  if (symbol === "fastMode" || symbol === "hsAuto" || symbol === "hsProtect") {
    try { if (typeof applyToSwitches === "function") applyToSwitches(); } catch(e) {}
  }
};

})();
